create or replace package body tbicds.PCK_ENCOUNTER_INTAKE is

/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

   /*get all the flags for an encounter*/
   procedure GetEncounterFlags(pi_vSessionID       in varchar2,
                              pi_vSessionClientIP in varchar2,
                              pi_nUserID          in number,
                              pi_vEncounterID     in varchar2,
                              po_nStatusCode      out number,
                              po_vStatusComment   out varchar2,
                              rs                  out Pck_Utl_Common.refCursor)
   is
      v_vSql                               varchar2(4000);
   begin
      -- default status to ok
      po_nStatusCode    := Pck_Utl_Common.c_nResultStatus_Success;
      po_vStatusComment := null;
      
      v_vSql := 'select t.*, it.topic, it.alternate_language as topic_alt, it.topic_image '
                || 'from tbicds.encounter_intake_flag t, tbicds.intake_topic it '
                || 'where t.encounter_id = :P0 '
                || 'and t.mid = it.mid '
                || 'and t.tid = it.tid '
                || 'order by t.mid, t.tid, t.flag_id';
      open rs for v_vSql using pi_vEncounterID;
  
   exception
      when others then
         po_nStatusCode    := Pck_Utl_Common.c_nResultStatus_Error;
         po_vStatusComment := 'PCK_ENCOUNTER_INTAKE.GetEncounterFlags(): ';
   end;

  /*
   get all the flags for an encounter intake
  */
   procedure GetEncounterIntakeFlags(pi_vSessionID         in varchar2,
                                    pi_vSessionClientIP   in varchar2,
                                    pi_nUserID            in number,
                                    pi_vEncounterID       in varchar2,
                                    pi_vEncounterIntakeID in number,
                                    po_nStatusCode        out number,
                                    po_vStatusComment     out varchar2,
                                    rs                    out Pck_Utl_Common.refCursor)
   is
      v_vSql                               varchar2(4000);
    v_lSQL long := '';
   begin
      -- default status to ok
      po_nStatusCode    := Pck_Utl_Common.c_nResultStatus_Success;
      po_vStatusComment := null;
      
      v_vSql := 'select t.*, it.topic, it.alternate_language as topic_alt, it.topic_image '
                || 'from tbicds.encounter_intake_flag t, tbicds.intake_topic it '
                || 'where t.encounter_id = :P0 '
                || 'and t.encounter_intake_id = :P1 '
                || 'and t.mid = it.mid '
                || 'and t.tid = it.tid '
                || 'order by t.mid, t.tid, t.flag_id';
      open rs for v_vSql using pi_vEncounterID, pi_vEncounterIntakeID;
  
   exception
      when others then
         po_nStatusCode    := Pck_Utl_Common.c_nResultStatus_Error;
         po_vStatusComment := 'PCK_ENCOUNTER_INTAKE.GetEncounterIntakeFlags(): ';
   end;

   /*
   Get the latest un-complete encounter for this module
   */
   procedure getIntakePatientID(pi_vSessionID         in varchar2,
                               pi_vSessionClientIP   in varchar2,
                               pi_nUserID            in number,
                               pi_vEncounterID       in varchar2,
                               pi_nEncounterIntakeID in number,
                               po_vPatientID         out varchar2,
                               po_nStatusCode        out number,
                               po_vStatusComment     out varchar2)
   is
      v_vSql                               varchar2(4000);
   begin
      po_vPatientID     := '';
      po_nStatusCode    := 0;
      po_vStatusComment := '';
      
      --get the latest un-complete encounter for this module
      begin
         --get the patient id from the
         --portal user id passed in
         v_vSql := 'select t.patient_id from tbicds.encounter t where t.encounter_id = :P0';
         execute immediate v_vSql into po_vPatientID using pi_vEncounterID;
      
      exception
         when others then
            po_vPatientID := '';
      end;
  
   exception
      when others then
         po_nStatusCode    := Pck_Utl_Common.c_nResultStatus_Error;
         po_vStatusComment := 'getIntakeAltLang(): ';
   end;

   /*
   Get alt language for the encounter
   */
   procedure getIntakeAltLang(pi_vSessionID         in varchar2,
                             pi_vSessionClientIP   in varchar2,
                             pi_nUserID            in number,
                             pi_vEncounterID       in varchar2,
                             pi_nEncounterIntakeID in number,
                             po_nOutAltLang        out number,
                             po_nStatusCode        out number,
                             po_vStatusComment     out varchar2)
   is
      v_vSql                               varchar2(4000);
   begin
      po_nOutAltLang    := 0;
      po_nStatusCode    := 0;
      po_vStatusComment := '';
      
      --get the latest un-complete encounter for this module
      begin
         --get the patient id from the
         --portal user id passed in
         v_vSql := 'select alternate_language from tbicds.encounter_intake where encounter_id = :P0 and encounter_intake_id = :P1';
         execute immediate v_vSql into po_nOutAltLang using pi_vEncounterID, pi_nEncounterIntakeID; 
      
      exception
         when others then
            po_nOutAltLang := 0;
      end;

   exception
      when others then
         po_nStatusCode    := Pck_Utl_Common.c_nResultStatus_Error;
         po_vStatusComment := 'getIntakeAltLang(): ';
   end;

   /*
   Tries to retrieve the encounter ID for the passed MID.
    If there is no encounter, it inserts a new encounter
    of type 7 (portal visit)
   */
   procedure GetEncounterIDFromModule(pi_vSessionID       in varchar2,
                                     pi_vSessionClientIP in varchar2,
                                     pi_nUserID          in number,
                                     pi_nModuleID        in number,
                                     pi_vPatientID       in varchar2,
                                     pi_nTreatmentID     in number,
                                     po_vEncounterID     out varchar2,
                                     po_nStatusCode      out number,
                                     po_vStatusComment   out varchar2)
   is
      v_vSql                               varchar2(4000);
      v_nSeq number;
      v_nCnt number;
      null_data EXCEPTION;
   begin
      po_nStatusCode    := 0; --0 = success
      po_vStatusComment := '';
   
      --get the encounter id for the latest
      --encounter intake matching the
      --mid passed in and that is not completed
      --this is so we can continue an assessment
      v_vSql := 'select max(t.encounter_id) '
                || 'from tbicds.encounter_intake t, tbicds.encounter e '
                || 'where t.encounter_id = e.encounter_id '
                || 'and e.patient_id = :P0 '
                || 'and t.mid = :P1 '
                || 'and nvl(complete, 0) < 1';
      execute immediate v_vSql into po_vEncounterID using pi_vPatientID, pi_nModuleID;
      
      --raise an exception
      if po_vEncounterID is null then
         RAISE null_data;
      end if;
      
   --if any of the above fail then we need to create a new:
   --   1. encounter (encounter type 99 = portal)
   --   2. encounter intake
   --get latest treatment id 
   exception
      when others then
         begin
           --encounter id if none, insert new encounter
         
           --get latest open encounter id, if not signed
           --then use it to create the encounter intake
           --just get the encounter for today if it exists
           v_vSql := 'select max(e.encounter_id) '
                     || 'from tbicds.encounter e '
                     || 'where e.patient_id = :P0 '
                     || 'and e.provider_signature_id is null ';
           execute immediate v_vSql into po_vEncounterID using pi_vPatientID;
         
         exception
            when others then
               po_vEncounterID := null;
         end; --encounter id if none, insert new encounter
 
         if (po_vEncounterID is null) then
            begin
               --does not have an open encounter for today
               --so create a new one based on the patient_id and
               --a sequence
               v_vSql := 'select tbicds.SEQ_ENCOUNTERID.nextval from dual';
               execute immediate v_vSql into v_nSeq;
               
               --new encounter id
               po_vEncounterID := mid(pi_vPatientID, 0, length(pi_vPatientID) - length(to_char(v_nSeq)))
                                  || to_char(v_nSeq);
               
               --insert a new encounter
               v_vSql := 'insert into tbicds.encounter '
                         || '(encounter_id, encounter_date, treatment_id, patient_id, encounter_type_id) '
                         || 'values (:P0, :P1, :P2, :P3, :P4)';
               execute immediate v_vSql using po_vEncounterID, sysdate, pi_nTreatmentID, pi_vPatientID, 99;
               commit;
           
            exception
               when others then
                  po_nStatusCode    := 1;
                  po_vStatusComment := 'PCK_ENCOUNTER_INTAKE.GetEncounterIDFromForModule(): ';
            end;
         end if; --if (v_strEncounterID is null) then
   end;

   /*
   get the the encounter_intake_id.  It checks for the existance
    of a record by checking the COMPLETE field.  If null, then
    there is no record.  In this case it inserts a new record. if
    complete exist and 0 then it returns the current id. 
   
    NOTE: It should never be called for a completed module 
   */
   procedure NewEncounterIntake(pi_vSessionID         in varchar2,
                               pi_vSessionClientIP   in varchar2,
                               pi_nUserID            in number,
                               pi_nModuleID          in number,
                               pi_vEncounterID       in varchar2,
                               pi_nAltLang           in number,
                               po_nEncounterIntakeID out number,
                               po_nStatusCode        out number,
                               po_vStatusComment     out varchar2)
   is
      v_nIncompleteCount number := 0;
      v_nIntakeCount     number := 0;
      v_nEiid            number := 0;
  
      v_vIncompleteCountSql constant varchar(32767) := 'select count(t.encounter_intake_id)
          from tbicds.encounter_intake t
          where t.encounter_id = :EncounterId
          and t.mid = :Mid
          and t.complete = 0';
      
      v_vIntakeCountSql constant varchar(32767) := 'select count(t.encounter_intake_id)
          from tbicds.encounter_intake t
          where t.encounter_id = :EncounterId';
      
      v_vMaxIdSql constant varchar(32767) := 'select max(t.encounter_intake_id)
         from tbicds.encounter_intake t
         where t.encounter_id = :EncounterId';
      
      v_vInsertSql constant varchar(32767) := 'insert into tbicds.encounter_intake t
          (t.encounter_id,
          t.encounter_intake_id,
          t.mid,
          t.intake_type,
          t.score,
          t.read_only,
          t.complete,
          t.alternate_language,
          t.review_encounter_id)
          values
          (:EncounterId,
          :EncounterIntakeId,
          :Mid,
          :IntakeType,
          :Score,
          :ReadOnly,
          :Complete,
          :AlternateLanguage,
          :ReviewEncounterId)';
      
      v_vIncompleteIdSql constant varchar(32767) := 'select t.encounter_intake_id
          from tbicds.encounter_intake t
          where t.encounter_id = :EncounterId
          and t.mid = :Mid
          and t.complete = 0';
   begin
      execute immediate v_vIncompleteCountSql into v_nIncompleteCount using pi_vEncounterID, pi_nModuleID;
  
      if v_nIncompleteCount = 0 then
         execute immediate v_vIntakeCountSql into v_nIntakeCount using pi_vEncounterID;
      
         if v_nIntakeCount = 0 then
            v_nEiid := 1;
         else
            execute immediate v_vMaxIdSql into v_nEiid using pi_vEncounterID;
            v_nEiid := v_nEiid + 1;
         end if;
         
         execute immediate v_vInsertSql using pi_vEncounterID, v_nEiid, pi_nModuleID, 1, 0, 0, 0, pi_nAltLang, '';
         
         po_nEncounterIntakeID := v_nEiid;
      else
         execute immediate v_vIncompleteIdSql into po_nEncounterIntakeID using pi_vEncounterID, pi_nModuleID;
      end if;
  
   exception
      when others then
         po_nEncounterIntakeID := -1;
         po_nStatusCode        := 1;
         po_vStatusComment     := 'PCK_ENCOUNTER_INTAKE.NewEncounterIntakeRS(): ';
   end;

   /*
   get a piece of a string given a delimeter and a position
   */
   function GetPiece(strData      in varchar2,
                    strDelimiter in varchar2,
                    nPosition    in NUMBER) return varchar2
   is
      strWorking varchar2(4000);
      strPiece   varchar2(400);
      pos        number;
      nPos2      number;
      nCount     number;
   begin
      --the piece to return
      strPiece := '';
      
      --set working to data
      strWorking := strData;
      
      --put a delimeter on the front for parsing if needed
      if substr(strWorking, 1, 1) != strDelimiter then
         strWorking := strDelimiter || strWorking;
      end if;

      --find the piece between the ~'s ie... "~piece~"
      pos    := -1;
      nCount := -1;
      
      while (pos != 0) loop
         pos := instr(strWorking, strDelimiter);
         
         if pos != 0 then
            --increment count
            nCount := nCount + 1;
            
            --found the piece...
            if nCount = nPosition then
               if (pos + 1 > length(strWorking) - 1) then
                  --nothing
                  strPiece := '';
               else
                  strWorking := substr(strWorking, pos + 1);
                  nPos2      := instr(strWorking, strDelimiter);
                  
                  if nPos2 = 0 then
                     --last piece
                     strPiece := strWorking;
                     return ltrim(rtrim(strPiece));
                  else
                     strPiece := substr(strWorking, 0, nPos2 - 1);
                     return ltrim(rtrim(strPiece));
                  end if;
               end if;
            else
               strWorking := substr(strWorking, pos + 1);
            end if;
         end if;
      end loop;
      
      return ltrim(rtrim(strPiece));
  
   exception
      when others then
         return '';
   end;

   /*
   get all Encounter Intake Responses count recordset
   */
   procedure GetResponsesCountRS(pi_vSessionID       in varchar2,
                                pi_vSessionClientIP in varchar2,
                                pi_nUserID          in number,
                                pi_vEncounterID     in varchar2,
                                pi_vMIDs            in varchar2,
                                po_nStatusCode      out number,
                                po_vStatusComment   out varchar2,
                                rs                  out RetRefCursor)
   is
      v_vSql                               varchar2(4000);
   begin
      po_nStatusCode    := 0; --0 = success
      po_vStatusComment := '';
      
      --open recordset
      v_vSql := 'select count(*) as count_resp, t.mid '
                || 'from tbicds.encounter_intake_responses t '
                || 'where t.encounter_id = :P0 '
                || 'and instr(:P1, '',''||t.mid||'','') > 0 '
                || 'group by t.mid '
                || 'order by t.mid ';
      open rs for v_vSql using pi_vEncounterID, ','||pi_vMIDs||',';
  
   exception
      when others then
         po_nStatusCode    := 1;
         po_vStatusComment := 'PCK_INTAKE.GetResponsesCountRS(): ';
   end;

   ------------------------------------------------------------------
   -- This SP will return a recordset with the data fields necessary
   -- to fill the education encounter list
   ------------------------------------------------------------------
   procedure GetIntakesForEducationRS(pi_vSessionID       in varchar2,
                                     pi_vSessionClientIP in varchar2,
                                     pi_nUserID          in number,
                                     pi_vPatientID       in varchar2,
                                     po_nStatusCode      out number,
                                     po_vStatusComment   out varchar2,
                                     rs                  out RetRefCursor)
   is
      v_vSql                               varchar2(4000);
   begin
      po_nStatusCode    := 0; --0 = success
      po_vStatusComment := '';
      
      --open recordset
      v_vSql := 'select m.mid, m.module as mid, e.encounter_date as edate '
                || 'from tbicds.intake_module m, tbicds.encounter e, tbicds.encounter_intake ei '
                || 'where e.patient_id = :P0 '
                || 'and ei.encounter_id = e.encounter_id '
                || 'and m.mid = ei.mid '
                || 'order by e.encounter_date ';
      open rs for v_vSql using pi_vPatientID;
  
   exception
      when others then
         po_nStatusCode    := 1;
         po_vStatusComment := 'PCK_INTAKE.GetIntakesForEducationRS(): ';
   end;
   
   /*get intakes for review recordset*/
   procedure GetIntakesForReviewRS(pi_vSessionID       in varchar2,
                                  pi_vSessionClientIP in varchar2,
                                  pi_nUserID          in number,
                                  pi_vPatientID       in varchar2,
                                  pi_vEncounterID      in varchar2,
                                  po_nStatusCode      out number,
                                  po_vStatusComment   out varchar2,
                                  rs                  out RetRefCursor)
   is
      v_vSql                               varchar2(4000);
      v_vPatientID  varchar2(50);
      v_nModalityID number;
   begin
      po_nStatusCode    := 0; --0 = success
      po_vStatusComment := '';
      
      v_vPatientID  := pi_vPatientID;
      
      --get modality id from encounter
      begin
         v_vSql := 'select e.encounter_type_id '
                   || 'from tbicds.encounter e '
                   || 'where e.encounter_id = :P0 '
                   || 'and e.patient_id = :P1 ';
         execute immediate v_vSql into v_nModalityID using pi_vEncounterID, v_vPatientID;
      exception
         when others then
            v_nModalityID := -1;
      end;
  
      v_vSql := 'select * from ('
                || 'select img.module_group_descr, m.module, p.*, imgm.sort_order '
                || 'from tbicds.patient_module p, '
                || 'tbicds.intake_module m, '
                || 'tbicds.intake_module_group img, '
                || 'tbicds.intake_module_group_mid imgm '
                || 'where m.mid = p.mid '
                || 'and img.module_group_id = p.module_group_id '
                || 'and imgm.module_group_id = p.module_group_id '
                || 'and imgm.mid = p.mid '
                || 'and p.patient_id = :P0 '
                --and p.status = 1
                || 'and p.module_group_id in (select module_group_id from tbicds.intake_module_group where stat_modality_id = :P1) '
                || ') t1 '
                || 'left join ('
                || 'select e.patient_id, i.* '
                || 'from tbicds.encounter e, tbicds.encounter_intake i '
                || 'where e.encounter_id = i.encounter_id '
                || 'and i.intake_group_id in (select module_group_id from tbicds.intake_module_group where stat_modality_id = :P2) '
                || 'and e.patient_id = :P3 '
                || ') t2 '
                || 'on t2.patient_id = t1.patient_id '
                || 'and t2.mid = t1.mid '
                || 'and t2.intake_group_id = t1.module_group_id '
                || 'order by t1.MODULE_GROUP_ID, t1.sort_order, t1.status desc, t1.date_completed ';
      open rs for v_vSql using v_vPatientID, v_nModalityID, v_nModalityID, v_vPatientID;
  
   exception
      when others then
         po_nStatusCode    := 1;
         po_vStatusComment := 'PCK_INTAKE.GetIntakesForReviewRS(): ';
   end;

end;
/

